/*
 * SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Copyright (c) UPMC, Lip6, SoC
 *         Nicolas Pouillon <nipo@ssji.net>, 2006-2007
 *
 * Maintainers: nipo
 */

#include <iostream>
#include <cstdlib>
#include <cstdio>

#include "mips32.h"
#include "mapping_table.h"
#include "gdbserver.h"
#include "vci_xcache_wrapper.h"
#include "vci_timer.h"
#include "vci_framebuffer.h"
#include "vci_fd_access.h"
#include "vci_ram.h"
#include "vci_multi_tty.h"
#include "vci_vgmn.h"
#include "vci_locks.h"
#include "vci_dma.h"

#include "segmentation.h"

#define FB_WIDTH 256
#define FB_HEIGHT 144

using namespace std;
using namespace sc_core;
using namespace soclib::caba;
using namespace soclib::common;

typedef VciParams<4,6,32,1,1,1,8,1,1,1> vci_param;

int _main(int argc, char *argv[])
{
  char * simulation_ncycles_var = getenv ("SIMULATION_N_CYCLES");
  char * simulation_ncpus_var = getenv ("SIMULATION_N_CPUS");
  long int n_cycles = 0, n_cpus = 0;

  /*
   * Check the parameters : param 1 is the name of the simulated application.
   */

  if (argc != 2)
  {
    cout << "[PLATFORM] Wrong number of arguments." << endl;
    cout << "Usage : ./simulation application_name" << endl;
    exit (-1);
  }

  /*
   * Simulation cycles
   */

  if (simulation_ncycles_var == NULL)
  {
    cout << "[PLATFORM] SIMULATION_N_CYCLES not defined." << endl;
    n_cycles = -1;
  }
  else
  {
    n_cycles = strtol(simulation_ncycles_var, (char **)NULL, 10);
    if (n_cycles == 0)
    {
      cout << "[PLATFORM] Invalid SIMULATION_N_CYCLES value, defaulting to infinite." << endl;
      n_cycles = -1;
    }
  }

  if (n_cycles == -1)
  {
    cout << "[PLATFORM] Infinite simulation." << endl;
  }
  else
  {
    cout << "[PLATFORM] Simulating " << n_cycles << " cycle(s)." << endl;
  }

  /*
   * Number of CPUs
   */

  if (simulation_ncpus_var == NULL)
  {
    cout << "[PLATFORM] SIMULATION_N_CPUS not defined." << endl;
    n_cpus = 1;
  }
  else
  {
    n_cpus = strtol(simulation_ncpus_var, (char **)NULL, 10);
    if (n_cpus == 0)
    {
      cout << "[PLATFORM] Invalid SIMULATION_N_CPUS value, defaulting to 1." << endl;
      n_cpus = 1;
    }
  }

  cout << "[PLATFORM] Using " << n_cpus << " CPU(s)." << endl;

  /*
   * Mapping table for the mips
   */

  MappingTable maptab(32, IntTab(8), IntTab(8), 0x00300000);

  maptab.add(Segment("reset",  RESET_BASE,  RESET_SIZE,   IntTab(0), false));
  maptab.add(Segment("excep",  EXCEP_BASE,  EXCEP_SIZE,   IntTab(0), false));
  maptab.add(Segment("memory", MEMORY_BASE, MEMORY_SIZE , IntTab(0), false));
  maptab.add(Segment("cmemory", CHANNEL_MEMORY_BASE, CHANNEL_MEMORY_SIZE , IntTab(1), false));
  maptab.add(Segment("tty",    TTY_BASE,    TTY_SIZE,     IntTab(2), false));
  maptab.add(Segment("timer",  TIMER_BASE,  TIMER_SIZE,   IntTab(3), false));
  maptab.add(Segment("fb",     FB_BASE,     FB_SIZE,      IntTab(4), false));
  maptab.add(Segment("fd",     FD_BASE,     FD_SIZE,      IntTab(5), false, true, IntTab(n_cpus)));
  maptab.add(Segment("dma", DMA_BASE, DMA_SIZE, IntTab(6), false));
  maptab.add(Segment("locks", LOCKS_BASE, LOCKS_SIZE, IntTab(7), false));


  /*
   * Signals
   */

  sc_clock signal_clk("signal_clk");
  sc_signal<bool> signal_resetn("signal_resetn");

  sc_signal<bool> signal_fd_it("signal_fd_it");
	sc_signal<bool> * signal_mips_it[n_cpus][6];
	VciSignals<vci_param> * signal_vci_m[n_cpus];

  for (int i = 0; i < n_cpus; i++)
  {
    char m_name[32];

    sprintf (m_name, "signal_vci_m%d", i);
    signal_vci_m[i] = new VciSignals<vci_param> (m_name);

    for (int j = 0; j < 6; j++)
    {
      char it_name[32];

      sprintf (it_name, "signal_mips%d_it%d", i, j);
      signal_mips_it[i][j] = new sc_signal<bool> (it_name);
    }
  }

  VciSignals<vci_param> signal_vci_vcimultiram0("signal_vci_vcimultiram0");
  VciSignals<vci_param> signal_vci_vcicram0("signal_vci_vcicram0");
  VciSignals<vci_param> signal_vci_tty("signal_vci_tty");
  VciSignals<vci_param> signal_vci_timer("signal_vci_timer");
  VciSignals<vci_param> signal_vci_framebuffer("signal_vci_framebuffer");
  VciSignals<vci_param> signal_vci_fd_i("signal_vci_fd_i");
  VciSignals<vci_param> signal_vci_fd_t("signal_vci_fd_t");
  soclib::caba::VciSignals<vci_param> signal_vci_dmai("signal_vci_dmai");
  soclib::caba::VciSignals<vci_param> signal_vci_dmat("signal_vci_dmat");
  soclib::caba::VciSignals<vci_param> signal_vci_locks("signal_vci_locks");


  /*
   * Processors
   */

  VciXcacheWrapper<vci_param, GdbServer<Mips32ElIss> > * cache[n_cpus];

  for (int i = 0; i < n_cpus; i++)
  {
    char cache_name[32];

    sprintf (cache_name, "CPU%d", i);
    cache[i] = new VciXcacheWrapper<vci_param, GdbServer<Mips32ElIss> > (cache_name, i, maptab, IntTab(i), 4, 1, 8, 4, 1, 8);

	  cache[i] -> p_clk (signal_clk);
	  cache[i] -> p_resetn (signal_resetn);
	  cache[i] -> p_vci (*signal_vci_m[i]);

    for (int j = 0; j < 6; j ++)
    {
	    cache[i] -> p_irq[j] (*signal_mips_it[i][j]); 
    }
  }

  /*
   * Memory
   */
  
	Loader loader(argv[1]);
	VciRam<vci_param> vcimultiram0("vcimultiram0", IntTab(0), maptab, loader);

  vcimultiram0 . p_clk(signal_clk);
	vcimultiram0 . p_resetn(signal_resetn);
	vcimultiram0 . p_vci(signal_vci_vcimultiram0);


  /*
   * Channel Memory
   */
  
	VciRam<vci_param> vcicram0("vcicmemory", IntTab(1), maptab, loader);

        vcicram0 . p_clk(signal_clk);
	vcicram0 . p_resetn(signal_resetn);
	vcicram0 . p_vci(signal_vci_vcicram0);



  /*
   * TTY
   */

  VciMultiTty<vci_param> vcitty("vcitty",	IntTab(2), maptab, "tty0", "tty1", NULL);

  vcitty . p_clk (signal_clk);
  vcitty . p_resetn (signal_resetn);
  vcitty . p_vci (signal_vci_tty);
  vcitty . p_irq[0] (*signal_mips_it[0][1]); 
  vcitty . p_irq[1] (*signal_mips_it[0][2]);

  /*
   * Timer
   */

	VciTimer<vci_param> vcitimer("vcittimer", IntTab(3), maptab, n_cpus);

  vcitimer . p_clk (signal_clk);
  vcitimer . p_resetn (signal_resetn);
  vcitimer . p_vci (signal_vci_timer);

  for (int i = 0; i < n_cpus; i += 1)
  {
    vcitimer . p_irq[i] (*signal_mips_it[i][0]); 
  }

  /*
<<<<<<< HEAD:top.cpp
   * Framebuffer
   */

  VciFrameBuffer<vci_param> vcifb("fb0", IntTab(4), maptab, FB_WIDTH, FB_HEIGHT);

  vcifb . p_clk (signal_clk);
  vcifb . p_resetn (signal_resetn);
  vcifb . p_vci (signal_vci_framebuffer);

  /*
   * FD
   */

	VciFdAccess<vci_param> vcifd("vcifd", maptab, IntTab(n_cpus), IntTab(5));

	vcifd . p_clk (signal_clk);
	vcifd . p_resetn (signal_resetn);
	vcifd . p_vci_initiator (signal_vci_fd_i);
	vcifd . p_vci_target (signal_vci_fd_t);
	vcifd . p_irq (signal_fd_it); 

/*
   * DMA
   */

	soclib::caba::VciDma<vci_param> vcidma("vcidma", maptab, IntTab(n_cpus+2), IntTab(6), (1<<(vci_param::K-1))); 


	vcidma.p_clk(signal_clk);
	vcidma.p_resetn(signal_resetn);
	vcidma.p_vci_target(signal_vci_dmat);
	vcidma.p_vci_initiator(signal_vci_dmai);
	vcidma.p_irq(*signal_mips_it[0][3]);

/*
   * LOCKS
   */

	soclib::caba::VciLocks<vci_param> vcilocks("vcilocks", IntTab(7), maptab); 


	vcilocks.p_clk(signal_clk);
	vcilocks.p_resetn(signal_resetn);
	vcilocks.p_vci(signal_vci_locks);


  /*
   * Network
   */

	VciVgmn<vci_param> vgmn("vgmn",maptab, n_cpus + 2, 8, 1, 8);

	vgmn . p_clk (signal_clk);
	vgmn . p_resetn (signal_resetn);

  for (int i = 0; i < n_cpus; i++)
  {
	  vgmn . p_to_initiator[i] (*signal_vci_m[i]);
  }

  vgmn . p_to_initiator[n_cpus] (signal_vci_fd_i);
  vgmn . p_to_initiator[n_cpus+1] (signal_vci_dmai);

	vgmn . p_to_target[0] (signal_vci_vcimultiram0);
	vgmn . p_to_target[1] (signal_vci_vcicram0);
  vgmn . p_to_target[2] (signal_vci_tty);
  vgmn . p_to_target[3] (signal_vci_timer);
  vgmn . p_to_target[4] (signal_vci_framebuffer);
	vgmn . p_to_target[5] (signal_vci_fd_t);
  vgmn . p_to_target[6] (signal_vci_dmat);
  vgmn . p_to_target[7] (signal_vci_locks);

  /*
   * Start of simulation
   */

	sc_start(sc_core::sc_time(0, SC_NS));
	signal_resetn = false;

	sc_start(sc_core::sc_time(1, SC_NS));
	signal_resetn = true;

	if (n_cycles == -1)
  {
    sc_start();
  }
  else
  {
    sc_start(sc_core::sc_time(n_cycles, SC_NS));
  }

	return EXIT_SUCCESS;
}

int sc_main (int argc, char *argv[])
{
	try
  {
		return _main(argc, argv);
	}
  catch (exception &e)
  {
		cout << e.what() << endl;
	}
  catch (...)
  {
		cout << "Unknown exception occured" << endl;
		throw;
	}

	return 1;
}
