/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <cstdlib>
#include <cstdio>

#include "mips32.h"
#include "mapping_table.h"
#include "gdbserver.h"
#include "vci_xcache_wrapper.h"
#include "vci_ram.h"
#include "vci_vgmn.h"

#include "segmentation.h"

using namespace std;
using namespace sc_core;
using namespace soclib::caba;
using namespace soclib::common;

typedef VciParams<4,6,32,1,1,1,8,1,1,1> vci_param;

int _main(int argc, char *argv[])
{
  char * simulation_ncycles_var = getenv ("SIMULATION_N_CYCLES");
  char * simulation_ncpus_var = getenv ("SIMULATION_N_CPUS");
  long int n_cycles = 0, n_cpus = 0;

  /*
   * Check the parameters : param 1 is the name of the simulated application.
   */

  if (argc != 2)
  {
    cout << "[PLATFORM] Wrong number of arguments." << endl;
    cout << "Usage : ./simulation application_name" << endl;
    exit (-1);
  }

  /*
   * Simulation cycles
   */

  if (simulation_ncycles_var == NULL)
  {
    cout << "[PLATFORM] SIMULATION_N_CYCLES not defined." << endl;
    n_cycles = -1;
  }
  else
  {
    n_cycles = strtol(simulation_ncycles_var, (char **)NULL, 10);
    if (n_cycles == 0)
    {
      cout << "[PLATFORM] Invalid SIMULATION_N_CYCLES value. ";
      cout << "Defaulting to infinite." << endl;
      n_cycles = -1;
    }
  }

  if (n_cycles == -1)
  {
    cout << "[PLATFORM] Infinite simulation." << endl;
  }
  else
  {
    cout << "[PLATFORM] Simulating " << n_cycles << " cycle(s)." << endl;
  }

  /*
   * Number of CPUs
   */

  if (simulation_ncpus_var == NULL)
  {
    cout << "[PLATFORM] SIMULATION_N_CPUS not defined." << endl;
    n_cpus = 1;
  }
  else
  {
    n_cpus = strtol(simulation_ncpus_var, (char **)NULL, 10);
    if (n_cpus == 0)
    {
      cout << "[PLATFORM] Invalid SIMULATION_N_CPUS value.";
      cout << "Defaulting to 1." << endl;
      n_cpus = 1;
    }
  }

  cout << "[PLATFORM] Using " << n_cpus << " CPU(s)." << endl;

  /*
   * Mapping table
   */

	MappingTable maptab(32, IntTab(8), IntTab(8), 0xC0000000);

	maptab . add (Segment ("reset",  RESET_BASE,  RESET_SIZE,  IntTab(0), false));
	maptab . add (Segment ("excep",  EXCEP_BASE,  EXCEP_SIZE,  IntTab(0), false));
	maptab . add (Segment ("memory", MEMORY_BASE, MEMORY_SIZE, IntTab(0), false));

  /*
   * Signals
   */

	sc_clock signal_clk ("signal_clk");
	sc_signal<bool> signal_resetn ("signal_resetn");

	sc_signal<bool> * signal_cpu_it[n_cpus][6];
	VciSignals<vci_param> * signal_cpu_i[n_cpus];

  for (int i = 0; i < n_cpus; i++)
  {
    char m_name[32];

    sprintf (m_name, "signal_cpu_i%d", i);
    signal_cpu_i[i] = new VciSignals<vci_param> (m_name);

    for (int j = 0; j < 6; j++)
    {
      char it_name[32];

      sprintf (it_name, "signal_cpu%d_it%d", i, j);
      signal_cpu_it[i][j] = new sc_signal<bool> (it_name);
    }
  }

	VciSignals<vci_param> signal_ram_t ("signal_ram_t");

  /*
   * Processors
   */

	VciXcacheWrapper<vci_param, GdbServer<Mips32ElIss> > * cache[n_cpus];

  for (int i = 0; i < n_cpus; i++)
  {
    char cache_name[32];

    sprintf (cache_name, "CPU%d", i);
    cache[i] = new VciXcacheWrapper<vci_param, GdbServer<Mips32ElIss> >
      (cache_name, i, maptab, IntTab(i), 4, 1, 8, 4, 1, 8);

	  cache[i] -> p_clk (signal_clk);
	  cache[i] -> p_resetn (signal_resetn);
	  cache[i] -> p_vci (*signal_cpu_i[i]);

    for (int j = 0; j < 6; j ++)
    {
	    cache[i] -> p_irq[j] (*signal_cpu_it[i][j]); 
    }
  }

  /*
   * Memory
   */
  
	Loader loader(argv[1]);
	VciRam<vci_param> ram ("ram", IntTab(0), maptab, loader);

  ram . p_clk (signal_clk);
	ram . p_resetn (signal_resetn);
	ram . p_vci (signal_ram_t);

  /*
   * Network
   */

	VciVgmn<vci_param> vgmn ("vgmn",maptab, n_cpus, 1, 1, 8);

	vgmn . p_clk (signal_clk);
	vgmn . p_resetn (signal_resetn);

  for (int i = 0; i < n_cpus; i++)
  {
	  vgmn . p_to_initiator[i] (*signal_cpu_i[i]);
  }

	vgmn . p_to_target[0] (signal_ram_t);

  /*
   * Start of simulation
   */

	sc_start (sc_core::sc_time(0, SC_NS));
	signal_resetn = false;

	sc_start (sc_core::sc_time(1, SC_NS));
	signal_resetn = true;

	if (n_cycles == -1)
  {
    sc_start();
  }
  else
  {
    sc_start(sc_core::sc_time(n_cycles, SC_NS));
  }

  /*
   * Delete the objects
   */

  for (int i = 0; i < n_cpus; i++)
  {
    delete cache[i];
    delete signal_cpu_i[i];

    for (int j = 0; j < 6; j++)
    {
      delete signal_cpu_it[i][j];
    }
  }

  /*
   * Return
   */

	return EXIT_SUCCESS;
}

int sc_main (int argc, char *argv[])
{
	try
  {
		return _main (argc, argv);
	}
  catch (exception &e)
  {
		cout << e.what() << endl;
	}
  catch (...)
  {
		cout << "Unknown exception occured" << endl;
		throw;
	}

	return 1;
}
