/*
 * SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Copyright (c) UPMC, Lip6, SoC
 *         Nicolas Pouillon <nipo@ssji.net>, 2006-2007
 *
 * Maintainers: nipo
 */

#include <iostream>
#include <cstdlib>
#include <cstdio>

#include "mips32.h"
#include "mapping_table.h"
#include "cc_id_table.h"
#include "gdbserver.h"
#include "vci_timer.h"
#include "vci_multi_tty.h"
#include "vci_cc_cache.h"
#include "vci_cc_ram.h"
#include "vci_vgmn.h"
#include "mailbox.h"

#include "segmentation.h"

#define DATA_BLOCK_SIZE 8 
#define CACHE_N_LINE 256

using namespace std;
using namespace sc_core;
using namespace soclib::caba;
using namespace soclib::common;

typedef VciParams<4,8,32,1,1,1,8,1,1,1> vci_param;
typedef	VciCcCache<vci_param, GdbServer<Mips32ElIss> > iss_t;

int _main(int argc, char *argv[])
{
  sc_trace_file * Tf = NULL;
  char * simulation_ncycles_var = getenv ("SIMULATION_N_CYCLES");
  char * simulation_ncpus_var = getenv ("SIMULATION_N_CPUS");
  long int n_cycles = 0, n_cpus = 0;

  /*
   * Check the parameters : param 1 is the name of the simulated application.
   */

  if (argc != 2)
  {
    cout << "[PLATFORM] Wrong number of arguments." << endl;
    cout << "Usage : ./simulation application_name" << endl;
    exit (-1);
  }

  /*
   * Simulation cycles
   */

  if (simulation_ncycles_var == NULL)
  {
    n_cycles = -1;
  }
  else
  {
    n_cycles = strtol(simulation_ncycles_var, (char **)NULL, 10);
    if (n_cycles == 0)
    {
      cout << "[PLATFORM] Invalid SIMULATION_N_CYCLES value, defaulting to infinite." << endl;
      n_cycles = -1;
    }
  }

  if (n_cycles == -1)
  {
    cout << "[PLATFORM] Infinite simulation." << endl;
  }
  else
  {
    cout << "[PLATFORM] Simulating " << n_cycles << " cycle(s)." << endl;
  }

  /*
   * Number of CPUs
   */

  if (simulation_ncpus_var == NULL)
  {
    cout << "[PLATFORM] SIMULATION_N_CPUS not defined." << endl;
    n_cpus = 1;
  }
  else
  {
    n_cpus = strtol(simulation_ncpus_var, (char **)NULL, 10);
    if (n_cpus == 0)
    {
      cout << "[PLATFORM] Invalid SIMULATION_N_CPUS value, defaulting to 1." << endl;
      n_cpus = 1;
    }
  }

  cout << "[PLATFORM] Using " << n_cpus << " CPU(s)." << endl;

  /*
   * Mapping table
   */

	CcIdTable * cctab = new CcIdTable();
	MappingTable maptab(32, IntTab(8), IntTab(8), 0xC0000000);

	maptab.add(Segment("reset",   RESET_BASE,   RESET_SIZE,   IntTab(0), true));
	maptab.add(Segment("excep",   EXCEP_BASE,   EXCEP_SIZE,   IntTab(0), true));
	maptab.add(Segment("memory",  MEMORY_BASE,  MEMORY_SIZE , IntTab(0), true));
  maptab.add(Segment("tty",     TTY_BASE,     TTY_SIZE,     IntTab(1), false));
  maptab.add(Segment("timer",   TIMER_BASE,   TIMER_SIZE,   IntTab(2), false));
  maptab.add(Segment("mailbox", MAILBOX_BASE, MAILBOX_SIZE, IntTab(3), false));

	for (int i = 0 ; i < n_cpus; i++)
	{
		char m_name[32];
		sprintf(m_name,"CPU%d",i);	
		maptab.add(Segment(m_name, CACHE_BASE + i * 0x1000000, CACHE_SIZE , IntTab(i + 4), false));
	}

  /*
   * Signals
   */

	sc_clock signal_clk("signal_clk");
	sc_signal<bool> signal_resetn("signal_resetn");

	sc_signal<bool> * signal_mips_it[n_cpus][6];
	VciSignals<vci_param> * signal_vci_m[n_cpus];
	VciSignals<vci_param> * signal_vci_inv_m[n_cpus];

  for (int i = 0; i < n_cpus; i++)
  {
    char m_name[32];

    sprintf (m_name, "signal_vci_m%d", i);
    signal_vci_m[i] = new VciSignals<vci_param> (m_name);

    sprintf (m_name, "signal_vci_inv_m%d", i);
    signal_vci_inv_m[i] = new VciSignals<vci_param> (m_name);

    for (int j = 0; j < 6; j++)
    {
      char it_name[32];

      sprintf (it_name, "signal_mips%d_it%d", i, j);
      signal_mips_it[i][j] = new sc_signal<bool> (it_name);
    }
  }

	VciSignals<vci_param> signal_vci_vcimultiram0("signal_vci_vcimultiram0");
  VciSignals<vci_param> signal_vci_tty("signal_vci_tty");
  VciSignals<vci_param> signal_vci_timer("signal_vci_timer");
	VciSignals<vci_param> signal_vci_inv_vcimultiram0("signal_vci_inv_vcimultiram0");

  VciSignals<vci_param> signal_vci_mailbox ("signal_vci_mailbox");
  sc_signal<bool> * signal_mailbox_it[32];

  for (int i = 0; i < 32; i += 1)
  {
    char mb_name[32];

    sprintf (mb_name, "signal_mailbox_it%d", i);
    signal_mailbox_it[i] = new sc_signal<bool> (mb_name);
  }

  /*
   * Registering Processor IDs
   */

	for (int k = 0; k < n_cpus; k ++)
	{
		cctab -> register_coherent_initiator(maptab.indexForId(IntTab(k)),IntTab(k+4));
	}

  /*
   * Registering Master device IDs
   */

  /*
   * Processors
   */

	iss_t * cache[n_cpus];

  for (int i = 0; i < n_cpus; i++)
  {
    char cache_name[32];

    sprintf (cache_name, "CPU%d", i);
	  cache[i] =  new iss_t (cache_name, IntTab(i), IntTab(i + 4), CACHE_N_LINE,
        DATA_BLOCK_SIZE, CACHE_N_LINE, DATA_BLOCK_SIZE, i, & maptab);

	  cache[i] -> p_clk (signal_clk);
	  cache[i] -> p_resetn (signal_resetn);
	  cache[i] -> p_i_vci (*signal_vci_m[i]);
	  cache[i] -> p_t_vci (*signal_vci_inv_m[i]);

    for (int j = 0; j < 6; j ++)
    {
	    cache[i] -> p_irq[j] (*signal_mips_it[i][j]); 
    }
  }

  /*
   * Memory
   */
  
	Loader loader(argv[1]);
	VciCcRam<vci_param> vcimultiram0("vcimultiram0",IntTab(n_cpus),
      IntTab(0), cctab, n_cpus, loader, DATA_BLOCK_SIZE, & maptab);

  vcimultiram0.p_clk(signal_clk);
	vcimultiram0.p_resetn(signal_resetn);
	vcimultiram0.p_t_vci(signal_vci_vcimultiram0);
	vcimultiram0.p_i_vci(signal_vci_inv_vcimultiram0);

  /*
   * TTY
   */

  VciMultiTty<vci_param> vcitty("vcitty",	IntTab(1), maptab, "tty0", "tty1", NULL);

  vcitty . p_clk (signal_clk);
  vcitty . p_resetn (signal_resetn);
  vcitty . p_vci (signal_vci_tty);
  vcitty . p_irq[0] (*signal_mips_it[0][1]); 
  vcitty . p_irq[1] (*signal_mips_it[0][2]); 

  /*
   * Timer
   */

	VciTimer<vci_param> vcitimer("vcittimer", IntTab(2), maptab, n_cpus);

  vcitimer . p_clk (signal_clk);
  vcitimer . p_resetn (signal_resetn);
  vcitimer . p_vci (signal_vci_timer);

  for (int i = 0; i < n_cpus; i += 1)
  {
    vcitimer . p_irq[i] (*signal_mips_it[i][0]); 
  }

  /*
   * Mailbox
   */

  mailbox<vci_param, 32> mbox("mailbox", IntTab(3), maptab);

  mbox . p_clk (signal_clk);
  mbox . p_resetn (signal_resetn);
  mbox . p_vci (signal_vci_mailbox);

  for (int i = 0; i < n_cpus; i += 1)
  {
    mbox . p_irq[i] (*signal_mips_it[i][5]); 
  }

  for (int i = n_cpus; i < 32; i += 1)
  {
    mbox . p_irq[i] (*signal_mailbox_it[i]); 
  }

  /*
   * Network
   */

	VciVgmn<vci_param> vgmn("vgmn",maptab, n_cpus + 1, n_cpus + 4, 1, 8);

	vgmn . p_clk (signal_clk);
	vgmn . p_resetn (signal_resetn);

  for (int i = 0; i < n_cpus; i++)
  {
	  vgmn . p_to_initiator[i] (*signal_vci_m[i]);
  }

  vgmn . p_to_initiator[n_cpus] (signal_vci_inv_vcimultiram0);

	vgmn . p_to_target[0] (signal_vci_vcimultiram0);
  vgmn . p_to_target[1] (signal_vci_tty);
  vgmn . p_to_target[2] (signal_vci_timer);
  vgmn . p_to_target[3] (signal_vci_mailbox);

  for (int i = 0; i < n_cpus; i++)
  {
	  vgmn . p_to_target[i + 4] (*signal_vci_inv_m[i]);
  }

  /*
   * Adding some signal tracing
   */

#if 0
  Tf = sc_create_vcd_trace_file("traces");

  // Base signals
  sc_trace (Tf, signal_clk, "clk");
  sc_trace (Tf, signal_resetn, "reset");
#endif

#if 0
  // Memory
  sc_trace (Tf, signal_vci_vcimultiram0 . cmdval, "signal_vci_vcimultiram0_cmdval");
  sc_trace (Tf, signal_vci_vcimultiram0 . cmdack, "signal_vci_vcimultiram0_cmdack");
  sc_trace (Tf, signal_vci_vcimultiram0 . rspval, "signal_vci_vcimultiram0_rspval");
  sc_trace (Tf, signal_vci_vcimultiram0 . rspack, "signal_vci_vcimultiram0_rspack");
  sc_trace (Tf, signal_vci_vcimultiram0 . rdata, "signal_vci_vcimultiram0_rdata");
  sc_trace (Tf, signal_vci_vcimultiram0 . address, "signal_vci_vcimultiram0_address");
  sc_trace (Tf, signal_vci_vcimultiram0 . srcid, "signal_vci_vcimultiram0_srcid");
#endif

#if 0
  // Mailbox
  sc_trace (Tf, signal_vci_mailbox . cmdval, "signal_vci_mailbox_cmdval");	
  sc_trace (Tf, signal_vci_mailbox . cmdack, "signal_vci_mailbox_cmdack");	
  sc_trace (Tf, signal_vci_mailbox . rspval, "signal_vci_mailbox_rspval");	
  sc_trace (Tf, signal_vci_mailbox . rspack, "signal_vci_mailbox_rspack");	
  sc_trace (Tf, signal_vci_mailbox . rdata, "signal_vci_mailbox_rdata");	
  sc_trace (Tf, signal_vci_mailbox . wdata, "signal_vci_mailbox_wdata");	
  sc_trace (Tf, signal_vci_mailbox . address,"signal_vci_mailbox_address");	
  sc_trace (Tf, signal_vci_mailbox . srcid, "signal_vci_mailbox_srcid");	

  for (int32_t i = 0; i < n_cpus; i += 1)
  {
    char name[32];

    sprintf (name,"signal_mailbox_it%d", i);	
    sc_trace (Tf, *signal_mips_it[i][5], name);
  }
#endif

#if 0
  // Caches
  for (int32_t i = 0; i < n_cpus; i += 1)
  {
    char name[32];

		sprintf (name, "signal_vci_m%d_cmdval", i);	
		sc_trace (Tf, signal_vci_m[i] -> cmdval, name);
		sprintf (name, "signal_vci_m%d_cmdack", i);	
		sc_trace (Tf, signal_vci_m[i] -> cmdack, name);
		sprintf (name, "signal_vci_m%d_rspval", i);	
		sc_trace (Tf, signal_vci_m[i] -> rspval, name);
		sprintf (name, "signal_vci_m%d_rspack", i);	
		sc_trace (Tf, signal_vci_m[i] -> rspack, name);
		sprintf (name, "signal_vci_m%d_rdata", i);	
		sc_trace (Tf, signal_vci_m[i] -> rdata, name);
		sprintf (name, "signal_vci_m%d_wdata", i);	
		sc_trace (Tf, signal_vci_m[i] -> wdata, name);
		sprintf (name, "signal_vci_m%d_address", i);	
		sc_trace (Tf, signal_vci_m[i] -> address, name);
		sprintf (name, "signal_vci_m%d_srcid", i);	
		sc_trace (Tf, signal_vci_m[i] -> srcid, name);
  }
#endif

  /*
   * Start of simulation
   */

	sc_start(sc_core::sc_time(0, SC_NS));
	signal_resetn = false;

	sc_start(sc_core::sc_time(1, SC_NS));
	signal_resetn = true;

	if (n_cycles == -1)
  {
    sc_start();
  }
  else
  {
    sc_start(sc_core::sc_time(n_cycles, SC_NS));
  }

  /*
   * Call to the destructors
   */

  for (int i = 0; i < 32; i++)
  {
    delete signal_mailbox_it[i];
  }
	
  for (int i = 0; i < n_cpus; i++)
  {
    delete signal_vci_m[i];
    delete signal_vci_inv_m[i];

    for (int j = 0; j < 6; j++)
    {
      delete signal_mips_it[i][j];
    }

    delete cache[i];
  }

	return EXIT_SUCCESS;
}

int sc_main (int argc, char *argv[])
{
	try
  {
		return _main(argc, argv);
	}
  catch (exception &e)
  {
		cout << e.what() << endl;
	}
  catch (...)
  {
		cout << "Unknown exception occured" << endl;
		throw;
	}

	return 1;
}
